/**
 * Upgram IA Metabox JavaScript
 * 
 * Gerencia a interação da metabox do Assistente de Conteúdo com IA
 * 
 * @package Upgram
 * @since 1.0.0
 */

// Log imediato para verificar se o arquivo está sendo executado
console.log('Upgram IA: Arquivo JavaScript carregado e executando...');
console.log('Upgram IA: jQuery disponível no início?', typeof jQuery !== 'undefined');
console.log('Upgram IA: window disponível?', typeof window !== 'undefined');

// Aguardar jQuery estar disponível
(function() {
    'use strict';
    
    function waitForJQuery() {
        if (typeof jQuery !== 'undefined') {
            console.log('Upgram IA: jQuery encontrado, inicializando...');
            initUpgramIA(jQuery);
        } else {
            console.log('Upgram IA: jQuery ainda não disponível, aguardando...');
            setTimeout(waitForJQuery, 100);
        }
    }
    
    // Tentar iniciar imediatamente
    if (typeof jQuery !== 'undefined') {
        console.log('Upgram IA: jQuery já disponível, iniciando imediatamente...');
        initUpgramIA(jQuery);
    } else {
        console.log('Upgram IA: jQuery não disponível, aguardando...');
        // Tentar quando DOM estiver pronto
        if (document.readyState === 'loading') {
            document.addEventListener('DOMContentLoaded', waitForJQuery);
        } else {
            waitForJQuery();
        }
    }
})();

function initUpgramIA($) {
    'use strict';
    
    console.log('Upgram IA: Script iniciado');
    console.log('Upgram IA: jQuery disponível?', typeof $ !== 'undefined');
    
    // Definir objeto fora do try para garantir que sempre exista
    let UpgramIA = null;
    
    try {
        console.log('Upgram IA: Definindo objeto UpgramIA...');
        
        UpgramIA = {
        
        /**
         * Inicializa o assistente
         */
        init: function() {
            console.log('Upgram IA: Inicializando...');
            
            if (typeof window.upgramIAData === 'undefined') {
                console.error('Upgram IA: window.upgramIAData não está definido!');
                return;
            }
            
            console.log('Upgram IA: Dados carregados, vinculando eventos...');
            this.bindEvents();
            console.log('Upgram IA: Inicialização completa');
        },
        
        /**
         * Vincula eventos
         */
        bindEvents: function() {
            const self = this;
            
            // Verificar se o botão existe
            const $generateBtn = $('#upgram-ia-generate-btn');
            console.log('Upgram IA: Procurando botão...', $generateBtn.length > 0 ? 'Encontrado' : 'NÃO ENCONTRADO');
            
            if ($generateBtn.length === 0) {
                console.error('Upgram IA: Botão #upgram-ia-generate-btn não encontrado!');
                return;
            }
            
            // Botão gerar
            $generateBtn.on('click', function(e) {
                e.preventDefault();
                console.log('Upgram IA: Botão clicado!');
                console.log('Upgram IA: Dados disponíveis?', typeof window.upgramIAData !== 'undefined');
                if (window.upgramIAData) {
                    console.log('Upgram IA: Dados:', window.upgramIAData);
                }
                self.generateContent();
            });
            
            console.log('Upgram IA: Evento vinculado ao botão');
            
            // Botão aplicar
            $('#upgram-ia-apply-btn').on('click', function(e) {
                e.preventDefault();
                self.applyContent();
            });
            
            // Botão copiar
            $('#upgram-ia-copy-btn').on('click', function(e) {
                e.preventDefault();
                self.copyContent();
            });
            
            // Botão descartar
            $('#upgram-ia-discard-btn').on('click', function(e) {
                e.preventDefault();
                self.discardResult();
            });
        },
        
        /**
         * Gera conteúdo com IA
         */
        generateContent: function() {
            console.log('Upgram IA: generateContent() chamado');
            const self = this;
            
            // Verificar se os dados estão disponíveis PRIMEIRO
            if (!window.upgramIAData || !window.upgramIAData.restUrl) {
                console.error('Upgram IA: Dados não encontrados!', window.upgramIAData);
                self.showError('Erro: Dados de configuração não encontrados. Recarregue a página.');
                return;
            }
            
            console.log('Upgram IA: Dados OK, continuando...');
            
            // Coletar dados
            const data = {
                post_id: parseInt(window.upgramIAData.postId),
                user_prompt: $('#upgram-ia-user-prompt').val() || '',
                use_current_content: $('#upgram-ia-use-current').is(':checked') ? true : false,
                tone: $('#upgram-ia-tone').val() || 'persuasive',
                keywords: $('#upgram-ia-keywords').val() || '',
                fields: []
            };
            
            // Coletar campos selecionados
            $('input[name="upgram-ia-fields[]"]:checked').each(function() {
                data.fields.push($(this).val());
            });
            
            // Validar
            if (data.fields.length === 0) {
                self.showError('Por favor, selecione pelo menos um campo para atualizar.');
                return;
            }
            
            // Validar post_id
            if (!data.post_id || data.post_id <= 0) {
                self.showError('Erro: ID do post inválido.');
                return;
            }
            
            // Mostrar loading
            $('#upgram-ia-generate-btn').prop('disabled', true);
            $('#upgram-ia-loading').show();
            $('#upgram-ia-result').hide();
            $('#upgram-ia-error').hide();
            
            // Log para debug
            console.log('Upgram IA: Enviando requisição', {
                url: window.upgramIAData.restUrl,
                data: data,
                nonce: window.upgramIAData.restNonce ? 'presente' : 'ausente'
            });
            
            // Fazer requisição usando jQuery AJAX
            console.log('Upgram IA: Usando jQuery AJAX');
            
            $.ajax({
                url: window.upgramIAData.restUrl,
                method: 'POST',
                beforeSend: function(xhr) {
                    xhr.setRequestHeader('X-WP-Nonce', window.upgramIAData.restNonce);
                    console.log('Upgram IA: Headers enviados', {
                        'X-WP-Nonce': window.upgramIAData.restNonce ? 'presente' : 'ausente'
                    });
                },
                data: data,
                dataType: 'json',
                timeout: 120000, // 2 minutos de timeout
                success: function(response) {
                    console.log('Upgram IA: Resposta recebida (jQuery)', response);
                    console.log('Upgram IA: Tipo da resposta', typeof response);
                    console.log('Upgram IA: Keys da resposta', Object.keys(response || {}));
                    self.handleSuccess(response);
                },
                error: function(xhr, status, error) {
                    console.error('Upgram IA: Erro na requisição (jQuery)', {
                        status: xhr.status,
                        statusText: xhr.statusText,
                        response: xhr.responseJSON || xhr.responseText,
                        error: error,
                        readyState: xhr.readyState
                    });
                    
                    // Log detalhado da resposta
                    if (xhr.responseJSON) {
                        console.error('Upgram IA: Erro JSON', xhr.responseJSON);
                    } else if (xhr.responseText) {
                        console.error('Upgram IA: Erro Text', xhr.responseText);
                    }
                    
                    self.handleError(xhr, status, error);
                },
                complete: function() {
                    console.log('Upgram IA: Requisição completa');
                    $('#upgram-ia-generate-btn').prop('disabled', false);
                    $('#upgram-ia-loading').hide();
                }
            });
        },
        
        /**
         * Trata erro do wp.apiFetch
         */
        handleApiFetchError: function(error) {
            console.error('Upgram IA: handleApiFetchError', error);
            
            let errorMessage = 'Não foi possível gerar o conteúdo. Verifique a conexão ou as configurações da IA.';
            
            if (error && error.message) {
                errorMessage = error.message;
            } else if (error && error.code) {
                errorMessage = 'Erro: ' + error.code;
                if (error.message) {
                    errorMessage += ' - ' + error.message;
                }
            } else if (error && error.data && error.data.message) {
                errorMessage = error.data.message;
            } else if (typeof error === 'string') {
                errorMessage = error;
            }
            
            // Log detalhado
            console.error('Upgram IA: Erro completo', {
                error: error,
                message: errorMessage
            });
            
            this.showError(errorMessage);
        },
        
        /**
         * Trata sucesso na geração
         */
        handleSuccess: function(response) {
            console.log('Upgram IA: handleSuccess chamado', response);
            
            // Verificar diferentes formatos de resposta
            let generatedContent = null;
            let responseData = null;
            
            if (response.success && response.data) {
                // Formato padrão WordPress REST API
                responseData = response.data;
                generatedContent = response.data.generated || response.data;
            } else if (response.generated) {
                // Formato direto
                responseData = response;
                generatedContent = response.generated;
            } else if (typeof response === 'string') {
                // String direta
                generatedContent = response;
                responseData = { generated: response };
            }
            
            if (generatedContent) {
                $('#upgram-ia-result-content').text(generatedContent);
                $('#upgram-ia-result').show();
                window.upgramIAGeneratedContent = responseData || { generated: generatedContent };
                console.log('Upgram IA: Conteúdo exibido com sucesso');
            } else {
                console.error('Upgram IA: Formato de resposta inesperado', response);
                this.showError('Não foi possível gerar o conteúdo. Resposta em formato inesperado. Verifique o console para mais detalhes.');
            }
        },
        
        /**
         * Trata erro na geração
         */
        handleError: function(xhr, status, error) {
            let errorMessage = 'Não foi possível gerar o conteúdo. Verifique a conexão ou as configurações da IA.';
            
            // Tentar extrair mensagem de erro da resposta
            if (xhr.responseJSON) {
                if (xhr.responseJSON.message) {
                    errorMessage = xhr.responseJSON.message;
                } else if (xhr.responseJSON.data && xhr.responseJSON.data.message) {
                    errorMessage = xhr.responseJSON.data.message;
                } else if (xhr.responseJSON.code) {
                    errorMessage = 'Erro: ' + xhr.responseJSON.code;
                    if (xhr.responseJSON.message) {
                        errorMessage += ' - ' + xhr.responseJSON.message;
                    }
                }
            } else if (xhr.responseText) {
                try {
                    const response = JSON.parse(xhr.responseText);
                    if (response.message) {
                        errorMessage = response.message;
                    } else if (response.data && response.data.message) {
                        errorMessage = response.data.message;
                    }
                } catch (e) {
                    // Se não conseguir parsear, verificar status HTTP
                    if (xhr.status === 0) {
                        errorMessage = 'Erro de conexão. Verifique sua internet.';
                    } else if (xhr.status === 403) {
                        errorMessage = 'Permissão negada. Verifique se você tem permissão para editar este post.';
                    } else if (xhr.status === 404) {
                        errorMessage = 'Endpoint não encontrado. Verifique se o plugin está ativo.';
                    } else {
                        errorMessage = 'Erro HTTP ' + xhr.status + ': ' + (error || 'Erro desconhecido');
                    }
                }
            }
            
            console.error('Upgram IA Error:', {
                status: xhr.status,
                statusText: xhr.statusText,
                response: xhr.responseJSON || xhr.responseText,
                error: error
            });
            
            this.showError(errorMessage);
        },
        
        /**
         * Mostra mensagem de erro
         */
        showError: function(message) {
            $('#upgram-ia-error').text(message).show();
            $('#upgram-ia-result').hide();
        },
        
        /**
         * Aplica o conteúdo gerado nos campos selecionados
         */
        applyContent: function() {
            if (!window.upgramIAGeneratedContent || !window.upgramIAGeneratedContent.generated) {
                this.showError('Nenhum conteúdo gerado para aplicar.');
                return;
            }
            
            const content = window.upgramIAGeneratedContent.generated;
            const parsed = window.upgramIAGeneratedContent.parsed || {};
            
            // Verificar se estamos no editor Gutenberg
            if (typeof wp !== 'undefined' && wp.data && wp.data.select('core/editor')) {
                this.applyToGutenberg(content, parsed);
            } else {
                this.applyToClassic(content, parsed);
            }
            
            // Mostrar mensagem de sucesso
            this.showSuccess('Conteúdo aplicado! Não esqueça de salvar o post.');
        },
        
        /**
         * Aplica conteúdo no editor Gutenberg
         */
        applyToGutenberg: function(content, parsed) {
            const dispatch = wp.data.dispatch('core/editor');
            const select = wp.data.select('core/editor');
            
            // Aplicar título se selecionado
            if ($('input[name="upgram-ia-fields[]"][value="post_title"]').is(':checked')) {
                const title = parsed.title || this.extractTitle(content) || select.getEditedPostAttribute('title');
                if (title) {
                    dispatch.editPost({ title: title });
                }
            }
            
            // Aplicar conteúdo se selecionado
            if ($('input[name="upgram-ia-fields[]"][value="post_content"]').is(':checked')) {
                const postContent = parsed.content || content;
                dispatch.editPost({ content: postContent });
            }
            
            // Aplicar excerto se selecionado
            if ($('input[name="upgram-ia-fields[]"][value="post_excerpt"]').is(':checked')) {
                const excerpt = parsed.excerpt || this.extractExcerpt(content) || '';
                dispatch.editPost({ excerpt: excerpt });
            }
        },
        
        /**
         * Aplica conteúdo no editor clássico
         */
        applyToClassic: function(content, parsed) {
            // Aplicar título se selecionado
            if ($('input[name="upgram-ia-fields[]"][value="post_title"]').is(':checked')) {
                const titleInput = $('#title');
                if (titleInput.length) {
                    const title = parsed.title || this.extractTitle(content) || titleInput.val();
                    titleInput.val(title).trigger('input');
                }
            }
            
            // Aplicar conteúdo se selecionado
            if ($('input[name="upgram-ia-fields[]"][value="post_content"]').is(':checked')) {
                const contentInput = $('#content');
                if (contentInput.length) {
                    const postContent = parsed.content || content;
                    contentInput.val(postContent);
                    
                    // Se estiver usando TinyMCE
                    if (typeof tinyMCE !== 'undefined' && tinyMCE.get('content')) {
                        tinyMCE.get('content').setContent(postContent);
                    }
                }
            }
            
            // Aplicar excerto se selecionado
            if ($('input[name="upgram-ia-fields[]"][value="post_excerpt"]').is(':checked')) {
                const excerptInput = $('#excerpt');
                if (excerptInput.length) {
                    const excerpt = parsed.excerpt || this.extractExcerpt(content) || '';
                    excerptInput.val(excerpt);
                }
            }
        },
        
        /**
         * Extrai título do conteúdo (tenta encontrar H1 ou primeira linha)
         */
        extractTitle: function(content) {
            // Tentar encontrar H1
            const h1Match = content.match(/<h1[^>]*>(.*?)<\/h1>/i);
            if (h1Match) {
                return h1Match[1].trim();
            }
            
            // Tentar primeira linha não vazia
            const lines = content.split('\n');
            for (let i = 0; i < lines.length; i++) {
                const line = lines[i].trim();
                if (line && !line.match(/^<[^>]+>/)) {
                    return line.substring(0, 100); // Limitar tamanho
                }
            }
            
            return null;
        },
        
        /**
         * Extrai excerto do conteúdo (primeiras palavras)
         */
        extractExcerpt: function(content) {
            // Remover HTML
            const text = $('<div>').html(content).text();
            // Pegar primeiras 150 palavras
            const words = text.split(/\s+/);
            return words.slice(0, 150).join(' ') + (words.length > 150 ? '...' : '');
        },
        
        /**
         * Copia conteúdo para área de transferência
         */
        copyContent: function() {
            if (!window.upgramIAGeneratedContent || !window.upgramIAGeneratedContent.generated) {
                this.showError('Nenhum conteúdo gerado para copiar.');
                return;
            }
            
            const content = window.upgramIAGeneratedContent.generated;
            const textarea = $('<textarea>').val(content).appendTo('body').select();
            
            try {
                document.execCommand('copy');
                textarea.remove();
                this.showSuccess('Conteúdo copiado para a área de transferência!');
            } catch (err) {
                textarea.remove();
                this.showError('Não foi possível copiar. Tente selecionar e copiar manualmente.');
            }
        },
        
        /**
         * Descarta resultado
         */
        discardResult: function() {
            $('#upgram-ia-result').hide();
            $('#upgram-ia-error').hide();
            window.upgramIAGeneratedContent = null;
        },
        
        /**
         * Mostra mensagem de sucesso
         */
        showSuccess: function(message) {
            const $notice = $('<div>')
                .addClass('notice notice-success is-dismissible')
                .css({
                    'position': 'fixed',
                    'top': '32px',
                    'right': '20px',
                    'z-index': '999999',
                    'max-width': '400px'
                })
                .html('<p>' + message + '</p>');
            
            $('body').append($notice);
            
            setTimeout(function() {
                $notice.fadeOut(function() {
                    $(this).remove();
                });
            }, 3000);
        }
    };
        
        // Verificar se UpgramIA está definido
        console.log('Upgram IA: Objeto UpgramIA definido!', typeof UpgramIA !== 'undefined' && UpgramIA !== null);
        if (UpgramIA) {
            console.log('Upgram IA: Métodos disponíveis:', Object.keys(UpgramIA));
            
            // Tornar global para debug
            window.UpgramIA = UpgramIA;
            console.log('Upgram IA: Objeto disponível globalmente como window.UpgramIA');
        } else {
            console.error('Upgram IA: ERRO - Objeto UpgramIA é null ou undefined!');
        }
    
        // Inicializar quando o DOM estiver pronto
        // Múltiplas formas de garantir que seja executado
        function upgramIAInit() {
            console.log('Upgram IA: upgramIAInit() chamado');
            console.log('Upgram IA: Verificando condições...', {
                jQuery: typeof jQuery !== 'undefined',
                upgramIAData: typeof window.upgramIAData !== 'undefined',
                UpgramIA: typeof UpgramIA !== 'undefined',
                windowUpgramIA: typeof window.UpgramIA !== 'undefined'
            });
            
            const targetIA = window.UpgramIA || UpgramIA;
            
            if (typeof jQuery !== 'undefined' && typeof window.upgramIAData !== 'undefined' && typeof targetIA !== 'undefined') {
                console.log('Upgram IA: Todas as condições OK, chamando init()...');
                try {
                    targetIA.init();
                    console.log('Upgram IA: init() executado sem erros');
                } catch (e) {
                    console.error('Upgram IA: Erro ao executar init()', e);
                    console.error('Upgram IA: Stack trace:', e.stack);
                }
            } else {
                console.warn('Upgram IA: Aguardando condições...', {
                    jQuery: typeof jQuery !== 'undefined',
                    upgramIAData: typeof window.upgramIAData !== 'undefined',
                    UpgramIA: typeof UpgramIA !== 'undefined',
                    windowUpgramIA: typeof window.UpgramIA !== 'undefined'
                });
                // Tentar novamente após um delay
                setTimeout(upgramIAInit, 500);
            }
        }
        
        // Tentar inicializar imediatamente se tudo estiver pronto
        if (typeof jQuery !== 'undefined' && typeof window.upgramIAData !== 'undefined' && typeof UpgramIA !== 'undefined') {
            console.log('Upgram IA: Tudo pronto, tentando inicializar imediatamente...');
            if (document.readyState === 'loading') {
                console.log('Upgram IA: DOM ainda carregando, aguardando DOMContentLoaded...');
                document.addEventListener('DOMContentLoaded', upgramIAInit);
            } else {
                console.log('Upgram IA: DOM pronto, inicializando via jQuery ready...');
                jQuery(document).ready(upgramIAInit);
            }
        } else {
            console.log('Upgram IA: Condições não atendidas, aguardando...');
            // Aguardar e tentar novamente
            if (document.readyState === 'loading') {
                document.addEventListener('DOMContentLoaded', upgramIAInit);
            } else {
                if (typeof jQuery !== 'undefined') {
                    jQuery(document).ready(upgramIAInit);
                } else {
                    setTimeout(upgramIAInit, 100);
                }
            }
        }
        
        // Fallback: tentar após um tempo
        setTimeout(function() {
            console.log('Upgram IA: Fallback timeout executado');
            const targetIA = window.UpgramIA;
            if (typeof targetIA !== 'undefined' && typeof window.upgramIAData !== 'undefined') {
                console.log('Upgram IA: Inicializando via timeout fallback...');
                try {
                    targetIA.init();
                } catch (e) {
                    console.error('Upgram IA: Erro no fallback', e);
                    console.error('Upgram IA: Stack trace:', e.stack);
                }
            } else {
                console.warn('Upgram IA: Fallback - condições não atendidas', {
                    UpgramIA: typeof window.UpgramIA !== 'undefined',
                    upgramIAData: typeof window.upgramIAData !== 'undefined'
                });
            }
        }, 2000);
    
    } catch (e) {
        console.error('Upgram IA: ERRO CRÍTICO no script!', e);
        console.error('Upgram IA: Stack trace:', e.stack);
        console.error('Upgram IA: Mensagem:', e.message);
        console.error('Upgram IA: Nome:', e.name);
    }
    
    // Garantir que o objeto esteja disponível mesmo se houver erro
    if (!window.UpgramIA && UpgramIA) {
        window.UpgramIA = UpgramIA;
        console.log('Upgram IA: Objeto definido no catch como fallback');
    }
    
    console.log('Upgram IA: Fim do script. window.UpgramIA disponível?', typeof window.UpgramIA !== 'undefined');
    
} // Fim da função initUpgramIA

